/*
 * Copyright (c) 2018-2020, NVIDIA CORPORATION.  All rights reserved. All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 */

#ifndef _CNVMTRANSPORTINK_HPP_
#define _CNVMTRANSPORTINK_HPP_

#include <vector>
#include <string>
#include "NvSIPLCommon.hpp"
#include "ModuleIF/CNvMCameraModule.hpp"
#include "devblk_cdi.h"
#include "utils/utils.hpp"

namespace nvsipl
{

//! Base Transport link class
class CNvMTransportLink
{
public:
    //! Transport link class manages the programming of ser-des pairs
    typedef struct {
        DevBlkCDIDevice* pSerCDIDevice;

        DevBlkCDIDevice* pDeserCDIDevice;

        //! Holding link index to be programmed by the object
        std::uint8_t ulinkIndex;

        //! Holding serializer native I2C address
        std::uint8_t uBrdcstSerAddr;

        //! Holding translation address for serializer
        std::uint8_t uSerAddr;

        CNvMCameraModule::ConnectionProperty moduleConnectionProperty;

        //! Flag indicating whether simulator mode has been enabled or not
        bool bEnableSimulator;

        //! Flag indicating whether slave mode has been enabled or not
        bool bSlave;

        //! Flag indicating whether the homogeneous camera support enabled or not
        bool m_groupInitProg;
    } LinkParams;

    //! This will do the init for a serializer + deserializer link
    virtual SIPLStatus Init(DevBlkCDIDevice* brdcstSerCDI, uint8_t linkMask, bool groupInitProg);

    //! Initialization steps to be done after the sensor has been configured
    virtual SIPLStatus PostSensorInit(DevBlkCDIDevice* brdcstSerCDI, uint8_t linkMask, bool groupInitProg);

    //! Additional initialization steps to be done after programming serializer, deserializer and sensors
    virtual SIPLStatus MiscInit();

    //! Start all the devices for the specific link
    virtual SIPLStatus Start();

    //! Stop all the devices for the specific link
    virtual SIPLStatus Stop();

    //! Reset all the devices for the specific link
    virtual SIPLStatus Reset();

    //! Deinit all the devices for the specific link
    SIPLStatus Deinit();

    uint8_t GetLinkIndex() {
        return m_oLinkParams.ulinkIndex;
    }

    virtual bool IsGMSL2() = 0;

    SIPLStatus SetConfig(LinkParams *params);

    virtual ~CNvMTransportLink()= default;

protected:
    //! Cached parameters
    LinkParams m_oLinkParams;
};

} // end of namespace nvsipl
#endif // _CNVMTRANSPORTINK_HPP_
