/* Copyright (c) 2018, NVIDIA CORPORATION.  All rights reserved.
 *
 * NVIDIA CORPORATION and its licensors retain all intellectual property
 * and proprietary rights in and to this software, related documentation
 * and any modifications thereto.  Any use, reproduction, disclosure or
 * distribution of this software and related documentation without an express
 * license agreement from NVIDIA CORPORATION is strictly prohibited.
 */

#ifndef _PARAMETER_PARSER_H_
#define _PARAMETER_PARSER_H_

#include <stdbool.h>

#include "context.h"
#include "nveeprom.h"
#include "nveeprom_spec.h"

/* The buffer size in bytes need to hold the data of the largest parameter */
/* This is the LSC data which is a 10x10 array of floats for each color channel */
#define MAX_PARAM_SIZE (sizeof(float)*4*10*10)

typedef struct {
    const char *name;
    NvEEPROMManagerParamId id;
    NvEEPROMParameterType type;
    uint32_t size;
    bool readOnly;
} ParameterSpec;

typedef struct {
    const ParameterSpec *spec;
    union {
        uint8_t  u8Value[MAX_PARAM_SIZE/sizeof(uint8_t)];
        uint16_t u16Value[MAX_PARAM_SIZE/sizeof(uint16_t)];
        uint32_t u32Value[MAX_PARAM_SIZE/sizeof(uint32_t)];
        uint64_t u64Value[MAX_PARAM_SIZE/sizeof(uint64_t)];
        float    floatValue[MAX_PARAM_SIZE/sizeof(float)];
        char     stringValue[MAX_PARAM_SIZE/sizeof(char)];
        uint8_t  binaryValue[MAX_PARAM_SIZE/sizeof(uint8_t)];
    };
} ParameterValue;

NvMediaStatus
ReadNvEEPROMManager(
    NvEEPROMManager *eepromManager,
    ParameterValue *parameters,
    uint32_t *size,
    bool readOnly);

NvMediaStatus
WriteNvEEPROMManager(
    NvEEPROMManager *eepromManager,
    const ParameterValue *parameters,
    uint32_t size);

/* Read a file and generate a parameter list from it */
NvMediaStatus
ReadParameterFile(
    FILE *parameterFile,
    ParameterValue *parameters,
    uint32_t *size);

/* Write the provided parameter list to a file */
NvMediaStatus
WriteParameterFile(
    FILE *parameterFile,
    const ParameterValue *parameters,
    uint32_t size,
    bool shortPrint);

void
GetReadOnlyParameters(
    const ParameterSpec **spec,
    uint32_t *count);

void
GetWritableParameters(
    const ParameterSpec **spec,
    uint32_t *count);

#endif /* _PARAMETER_PARSER_H_ */
