/*
 * Copyright (c) 2018-2020, NVIDIA CORPORATION.  All rights reserved. All
 * information contained herein is proprietary and confidential to NVIDIA
 * Corporation.  Any use, reproduction, or disclosure without the written
 * permission of NVIDIA Corporation is prohibited.
 */

#ifndef _CNVMSENSOR_HPP_
#define _CNVMSENSOR_HPP_

#include "NvSIPLPlatformCfg.hpp"
#include "devblk_cdi.h"

#include "CNvMDevice.hpp"
#include "utils/utils.hpp"

namespace nvsipl
{

//! Base sensor class
class CNvMSensor : public CNvMDevice
{
public:
    typedef struct {
        void *handle; //Handle passed from Sensor to EEPROM
        uint32_t size;
    } EEPROMRequestHandle;

    //! Constructor
    CNvMSensor();

    //! Destructor
    virtual ~CNvMSensor() = default;

    virtual SIPLStatus SetConfig(const SensorInfo *sensorInfo, DeviceParams *params);

    //! Set the pipeline index for the sensor
    virtual void SetPipelineIndex(uint32_t);

    //! Get sensor's pipeline index
    virtual uint32_t GetPipelineIndex();

    //! Get width
    virtual uint32_t GetWidth() const;

    //! Get height
    virtual uint32_t GetHeight() const;

    //! Get number of top embedded lines
    virtual uint32_t GetEmbLinesTop() const;

    //! Get number of bottom embedded lines
    virtual uint32_t GetEmbLinesBot() const;

    //! Get embedded data type
    virtual bool GetEmbDataType() const;

    //! Get input format type
    virtual NvMediaICPInputFormat GetInputFormat() const;

    //! Get surface type
    virtual NvMediaSurfaceType GetSurfaceType() const;

    //! Get frame rate
    virtual float GetFrameRate() const;

    //! Get Enable external sync
    virtual bool GetEnableExtSync() const;

    //! Get Enable TPG
    virtual bool GetEnableTPG() const;

    //! Get Pattern Mode
    virtual uint32_t GetPatternMode() const;

    //! Get Sensor Identifier
    virtual std::string GetSensorDescription() const;

    virtual EEPROMRequestHandle GetEEPROMRequestHandle() final;

    virtual void SetEEPROMRequestHandle(void *handle, uint32_t size) final;

    void SetDriverHandle(DevBlkCDIDeviceDriver *driver) {m_pCDIDriver = driver;}
    void SetDriverContext(std::unique_ptr<DriverContext> context) {m_upDrvContext = std::move(context);}

protected:

    //! Set input format property
    SIPLStatus SetInputFormatProperty();

    //! Set surface type
    SIPLStatus SetSurfaceType();

    //! Holding pipeline index for the sensor
    uint32_t m_pipelineIndex;

    //! Holding number of top embedded lines
    uint32_t m_embLinesTop;

    //! Holding number of bottom embedded lines
    uint32_t m_embLinesBot;

    //! Holding embedded data type
    bool m_bEmbDataType;

    //! Holding width
    uint32_t m_width;

    //! Holding height
    uint32_t m_height;

    //! Holding ICP input format
    NvMediaICPInputFormat m_eInputFormat;

    //! Holding pixel order
    uint32_t m_ePixelOrder;

    //! Holding surface type
    NvMediaSurfaceType m_eSurfType;

    //! Holding frame rate
    float m_frameRate;

    //! Flag indicating whether the sensor is in trigger mode
    bool m_bEnableExtSync;

    //! Holding enable TPG
    bool m_bEnabletpg;

    //! Holding sensor description
    std::string m_sensorDescription;

    EEPROMRequestHandle m_EEPROMRequestHandle;

    //! Holding Test Pattern Mode
    uint32_t m_patternMode;
};

} // end of namespace nvsipl
#endif //_CNVMSENSOR_HPP_
